<?php
/**
 * Codice Fiscale.
 * Classe per la rappresentazione ed il controllo di validit
 *	di un codice fiscale.
 *
 * @since PHP 4.0.0 (3.0?)
 * @author Stefano Luchetti <stefano.luchetti@kronositalia.com>
 * @see http://www.kronositalia.com
 */
 
class CodiceFiscale
{
	// {{{ properties
  /**
   * Variabile d'istanza per l'allocazione della stringa $args['codiceFiscale']	
	*
   * @var string
   * @access private
   */
	
	private $codiceFiscale;
	
	// }}}
	
	// {{{ Costruttore e Distruttore
	/**
   * Costruttore
   *
   * @param string $args['cartaDiCredito'] il numero della carta di credito
   * @return non applicabile
   * @access public
   */
	public function __construct($args)
	{
		$this -> setCodiceFiscale($args['codiceFiscale']);	
	}

	/**
   * Distruttore
   *
   * @param nessuno
   * @return non applicabile
   * @access public
   */
	
	public function __destruct()
	{
	
	}
	// }}}
	
	
	// {{{ getCodiceFiscale()
	/**
   * Accessor method della variabile d'istanza $codiceFiscale
   *
   * @param nessuno
   * @return la variabile d'istanza $codiceFiscale
   * @access public
   */
	
	public function getCodiceFiscale()
	{
		return $this -> codiceFiscale;
	}
	// }}}
	
	// {{{ setCodiceFiscale()
	/**
   * Mutator method della variabile d'istanza $codiceFiscale
   *
   * @param string $codiceFiscale
   * @return void
   * @access public
   */
	
	public function setCodiceFiscale($codiceFiscale)
	{
		if (!$codiceFiscale) 
		{
			trigger_error("Istanziare il controllore di codice fiscale quando si ha un codice da controllare!",E_USER_ERROR);
		}
		$this -> codiceFiscale = strtoupper(trim($codiceFiscale));
	}
	// }}}
	
	// {{{ controllaErrori()
	/**
   * Metodo che implementa l'algoritmo di controllo del codice fiscale:
	*
	* (1)	controllo di lunghezza pari a 16 caratteri [codice d'errore: 2]
	* (2)	controllo di corrispondenza del pattern LLLLLLCCLCCLCCCL con 
	*		lettere (L) e cifre (C) [codice d'errore: 3]
	* (3)	controllo di corrispondenza tra la parte anagrafica (LLLLLLCCLCCLCCC) 
	*		e ultima lettera di controllo [codice d'errore: 3]
	*	
   * @param string $codiceFiscale
   * @return int il codice di errore del problema riscontrato
   * @access public
   */

	public function controllaErrori()
	{
		$cf = $this -> getCodiceFiscale();

		/*
			Restituisce una stringa di 3 bit aventi il seguente significato:
			Bit nro 1: Se 1 => Errore di lunghezza
			Bit nro 2: Se 1 => Pattern LLLLLLCCLCCLCCCL non rispettato (L = lettera, C = cifra)
			Bit nro 3: Se 1 => Carattere di controllo X(LLLLLLCCLCCLCCCX) non coerente
		 */

		try 
		{

			if( $cf == '' )  return; // Non controllo il nulla
			
			if( strlen($cf) != 16 )
			{
				return "100";
				//throw new CodiceFiscaleException("Il codice fiscale non rispetta i 16 caratteri previsti.",1);
			}	
	
			// [a-zA-Z]{6}[0-9]{2}[a-zA-Z][0-9]{2}[a-zA-Z][0-9]{3}[a-zA-Z]
			$cf = strtoupper($cf);
			if( ! ereg("^[A-Z]{6}[0-9]{2}[A-Z][0-9]{2}[A-Z][0-9]{3}[A-Z]$", $cf) ){
				return "010";
				//throw new CodiceFiscaleException("Il codice fiscale non rispetta la sequenza di caratteri prevista.",2);
			}
			
			$s = 0;
			for( $i = 1; $i <= 13; $i += 2 )
			{
				$c = $cf[$i];
				if( '0' <= $c && $c <= '9' )
				{
					$s += ord($c) - ord('0');
				}		
				else
				{
					$s += ord($c) - ord('A');
				}	
			}
			
			for( $i = 0; $i <= 14; $i += 2 )
			{
				$c = $cf[$i];
				
				switch( $c )
				{
					case '0':  $s += 1;  break;
					case '1':  $s += 0;  break;
					case '2':  $s += 5;  break;
					case '3':  $s += 7;  break;
					case '4':  $s += 9;  break;
					case '5':  $s += 13;  break;
					case '6':  $s += 15;  break;
					case '7':  $s += 17;  break;
					case '8':  $s += 19;  break;
					case '9':  $s += 21;  break;
					case 'A':  $s += 1;  break;
					case 'B':  $s += 0;  break;
					case 'C':  $s += 5;  break;
					case 'D':  $s += 7;  break;
					case 'E':  $s += 9;  break;
					case 'F':  $s += 13;  break;
					case 'G':  $s += 15;  break;
					case 'H':  $s += 17;  break;
					case 'I':  $s += 19;  break;
					case 'J':  $s += 21;  break;
					case 'K':  $s += 2;  break;
					case 'L':  $s += 4;  break;
					case 'M':  $s += 18;  break;
					case 'N':  $s += 20;  break;
					case 'O':  $s += 11;  break;
					case 'P':  $s += 3;  break;
					case 'Q':  $s += 6;  break;
					case 'R':  $s += 8;  break;
					case 'S':  $s += 12;  break;
					case 'T':  $s += 14;  break;
					case 'U':  $s += 16;  break;
					case 'V':  $s += 10;  break;
					case 'W':  $s += 22;  break;
					case 'X':  $s += 25;  break;
					case 'Y':  $s += 24;  break;
					case 'Z':  $s += 23;  break;
				}
			}
	
			if( chr($s%26 + ord('A')) != $cf[15] )
			{
				return "001";
				//throw new CodiceFiscaleException("Codice fiscale e codice di controllo sono incompatibili.",3);
			}
			
			return "000";
		
		}
		catch(CodiceFiscaleException $cfe)
		{
			echo $cfe;
		}
		catch(Exception $e)
		{
			echo($e);
		}

	}
	// }}}
}

class CodiceFiscaleException extends Exception
{

	public function __construct($message, $code = 0) {
		// Invocazione costruttore della Base Class
		parent::__construct($message, $code);
		// Far altro??
	}

	public function __toString() {
	  return __CLASS__ . ": [{$this->code}]: {$this->message}\n";
	}
	
} 


class CodiceFiscaleAvanzato extends CodiceFiscale
{

	private $dataDiNascita;
	private $nome;
	private $cognome;
	private $arrayMesi;
	
	public function __construct($args)
	{

		/* 
			Invoco il costruttore della classe padre ...  la prima - essenziale -
			cosa da fare.
		 */
		parent::__construct($args);

		$this -> setNome($args['nome']);
		$this -> setCognome($args['cognome']);
		$this -> setDataDiNascita($args['dataDiNascita']);
		
		$this -> arrayMesi = array	( "01" => "A",	"02" => "B", "03" => "C",
											  "04" => "D", "05" => "E", "06" => "H",
											  "07" => "L",	"08" => "M", "09" => "P",
		     								  "10" => "R", "11" => "S", "12" => "T" ); 
		
	}
	
	public function __destruct()
	{
	
	}


	public function getDataDiNascita()
	{
		return $this -> dataDiNascita;
	}

	public function setDataDiNascita($dataDiNascita)
	{
		$this -> dataDiNascita = $dataDiNascita;
	}


	public function getNome()
	{
		return $this -> nome;
	}

	public function setNome($nome)
	{
		$this -> nome = strtoupper($nome);
	}


	public function getCognome()
	{
		return $this -> cognome;
	}

	public function setCognome($cognome)
	{
		$this -> cognome = strtoupper($cognome);
	}


	public function controllaDataDiNascita()
	{

		$d = explode("-",$this -> getDataDiNascita());
		$AA = substr($d[0],2,2); $M = $this -> arrayMesi[$d[1]]; $GG = $d[2];

		$fCase = $AA.$M.($GG+40); // caso femmina
		$mCase = $AA.$M.($GG); // caso maschio
		
		$subStrData = strtoupper(substr($this -> getCodiceFiscale(),6,5));
		
		return ($subStrData === $mCase || $subStrData === $fCase)?"0":"1";

	}
	
	private function controllaOnomastica()
	{
		$codicefiscale = $this -> getCodiceFiscale();
		$podc = substr($codicefiscale,0,6); // parte onomastica di confronto
		
		$nome = $this -> getNome();
		$cognome = $this -> getCognome();
		
		$n = str_split(str_replace(" ","",strtoupper($nome)));
		$c = str_split(str_replace(" ","",strtoupper($cognome)));
		
		$vocali = array("A","E","I","O","U");
		$consonanti = array("B","C","D","F","G","H","J","K","L","M","N","P","Q","R","S","T","V","W","X","Y","Z");
		
		$cc = array(); // consonanti cognome
		$vc = array(); // vocali cognome
		foreach( $c as $lettera )
		{
			if (in_array($lettera,$consonanti)){array_push($cc,$lettera);}
			if (in_array($lettera,$vocali))    {array_push($vc,$lettera);}
		}
		
		$cn = array(); // consonanti nome
		$vn = array(); // vocali nome
		foreach( $n as $lettera )
		{
			if (in_array($lettera,$consonanti)){array_push($cn,$lettera);}
			if (in_array($lettera,$vocali))    {array_push($vn,$lettera);}
		}
		
		$pc = (isset($cc[2]))?$cc[0].$cc[1].$cc[2]:""; // parte cognome
		for ($i=0;strlen($pc) < 3 && isset($cc[$i]);) { $pc .= $cc[$i++];}
		for ($i=0;strlen($pc) < 3 && isset($vc[$i]);) { $pc .= $vc[$i++];}
		for (;strlen($pc) < 3;) { $pc .= "X";}
		
		$pn = (isset($cn[3]))?$cn[0].$cn[2].$cn[3]:""; // parte nome
		for ($i=0;strlen($pn) < 3 && isset($cn[$i]);) { $pn .= $cn[$i++];}
		for ($i=0;strlen($pn) < 3 && isset($vn[$i]);) { $pn .= $vn[$i++];}
		for (;strlen($pn) < 3;) { $pn .= "X";}
		
		$po = $pc.$pn; // parte onomastica = parte nome + parte cognome
		
		return ($po === $podc)?"0":"1";

	}

	/*
		Overridding del metodo ereditato .... deve fare qualche cosa in pi	
	 */
	public function controllaErrori()
	{
		/*
			Restituisce una stringa di 5 bit aventi il seguente significato:
			Bit nro 1: Se 1 => Errore di lunghezza
			Bit nro 2: Se 1 => Pattern LLLLLLCCLCCLCCCL non rispettato (L = lettera, C = cifra)
			Bit nro 3: Se 1 => Carattere di controllo X(LLLLLLCCLCCLCCCX) non coerente
			Bit nro 4: Se 1 => "Data di nascita di riferimento" e "data di nascita 
						  			 nel codice fiscale" non coerenti
			Bit nro 5: Se 1 => "Nome-Cognome" e "parte onomastica del codice fiscale" non coerenti
		 */
		if (!$this -> getCodiceFiscale()) { return "00000";}
		$controllo  = parent :: controllaErrori();
		$controllo .= $this -> controllaDataDiNascita();
		$controllo .= $this -> controllaOnomastica();
		return $controllo;
	}


}




?>